<?php
namespace KeyDesign\Compatibility;

use KeyDesign\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'ElementsKit_Lite' ) ) {
	return;
}

class KeyDesign_ElementsKit_Lite {

	private static $instance;

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {

		// Header builder markup
		add_action( 'elementskit/template/before_header', [ $this, 'header_builder_before_header_markup' ] );
		add_action( 'elementskit/template/after_header', [ $this, 'header_builder_after_header_markup' ] );

		// Footer builder markup
		add_action( 'elementskit/template/before_footer', [ $this, 'footer_builder_before_footer_markup' ] );
		add_action( 'elementskit/template/after_footer', [ $this, 'footer_builder_after_footer_markup' ] );

		// Admin menu actions
		add_action( 'admin_menu', [ $this, 'admin_menu_actions' ], -1 );

		// Remove dashboard widgets
		add_action( 'wp_dashboard_setup', [ $this, 'remove_dashboard_widgets' ] );

		// Set onboarding status
		add_action( 'init', [ $this, 'onboarded_status' ] );

		// Dismiss banners
		add_filter( 'elementskit/license/hide_banner', '__return_true' );

		// Overwrite widget controls
		require_once KEYDESIGN_COMPATIBILITY_PATH . '/elementskit/widget-controls.php';

		// Update ElementsKit options array
		$this->update_elementskit_options();

		// Admin menu
		add_action( 'admin_menu', [ $this, 'elementskit_menu' ], 100000 );

		// Add body class for header footer builder
		add_filter( 'elementor/editor/after_enqueue_styles', [ $this, 'elementskit_header_footer_body_class' ] );

		// Disable ElementsKit Pro update notifications
		$this->disable_update_notifications();

	}

	public function header_builder_before_header_markup() {
		echo '<div id="page" class="site"><header id="site-header" ' . keydesign_site_header_class( '' , $echo = false ) . '><div class="site-header-wrapper">';
	}

	public function header_builder_after_header_markup() {
		echo '</div></header>
			<div id="content" class="site-content">';
	}

	public function footer_builder_before_footer_markup() {
		echo '</div><!-- #content -->';
		do_action( 'keydesign_content_after' );
		do_action( 'keydesign_footer_before' );
		echo '<footer id="site-footer" ' . keydesign_site_footer_class( '' , $echo = false ) . ' role="contentinfo">';
		do_action( 'keydesign_footer_top' );
	}

	public function footer_builder_after_footer_markup() {
		do_action( 'keydesign_footer_bottom' );
		echo '</footer>';
		do_action( 'keydesign_footer_after' );
		echo '</div><!-- #page -->';
		do_action( 'keydesign_body_bottom' );
	}

	public function admin_menu_actions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		add_menu_page(
			__( 'Headers', 'keydesign-framework' ),
			__( 'Headers', 'keydesign-framework' ),
			'manage_options',
			'edit.php?post_type=elementskit_template&elementskit_type_filter=header',
			'',
			'dashicons-align-center',
			30
		);

		add_menu_page(
			__( 'Footers', 'keydesign-framework' ),
			__( 'Footers', 'keydesign-framework' ),
			'manage_options',
			'edit.php?post_type=elementskit_template&elementskit_type_filter=footer',
			'',
			'dashicons-align-center',
			35
		);
	}

	public function remove_dashboard_widgets() {
		remove_meta_box( 'wpmet-stories', 'dashboard', 'normal' );
	}

	public function onboarded_status() {
		add_option('default_ekit_settings', 1);

		if ( get_option('default_ekit_settings') == 1 ) {

			update_option( 'elements_kit_onboard_status', 'onboarded' );

			$elemkit_options = Utils::get_elementskit_options();

			$elemkit_options[ 'module_list' ][ 'elementskit-icon-pack' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'header-footer' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'megamenu' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'onepage-scroll' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'sticky-content' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'copy-paste-cross-domain' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'advanced-tooltip' ][ 'status' ] = 'active';
			$elemkit_options[ 'module_list' ][ 'masking' ][ 'status' ] = 'active';

			$elemkit_options[ 'module_list' ][ 'widget-builder' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'facebook-messenger' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'conditional-content' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'pro-form-reset-button' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'google_sheet_for_elementor_pro_form' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'particles' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'module_list' ][ 'parallax' ][ 'status' ] = 'active';

			$elemkit_options[ 'widget_list' ][ 'drop-caps' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'dual-button' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'tablepress' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'back-to-top' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'advanced-accordion' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'chart' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'table' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'unfold' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'whatsapp' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'team-slider' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'flip-box' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'content-ticker' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'coupon-code' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'category-list' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'post-grid' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'post-tab' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'vertical-menu' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'caldera-forms' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'we-forms' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'wp-forms' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'ninja-forms' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'fluent-forms' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'twitter-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'instagram-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'behance-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'dribble-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'facebook-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'pinterest-feed' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'zoom' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'facebook-review' ][ 'status' ] = 'inactive';
			$elemkit_options[ 'widget_list' ][ 'yelp' ][ 'status' ] = 'inactive';

			update_option( 'elementskit_options', $elemkit_options );
			update_option( 'default_ekit_settings', 0 );
		}
	}

	public static function update_elementskit_options() {
		$elemkit_options = Utils::get_elementskit_options();
		$api_code = Utils::get_option( 'google_maps_api' );
		
		if ( $api_code != '' ) {
			if ( !isset( $elemkit_options[ 'user_data' ][ 'google_map' ][ 'api_key' ] ) || $elemkit_options[ 'user_data' ][ 'google_map' ][ 'api_key' ] != $api_code ) {
				$elemkit_options[ 'user_data' ][ 'google_map' ][ 'api_key' ] = $api_code;
			}
		}

		if ( !isset( $elemkit_options[ 'settings' ][ 'ekit_user_consent_for_banner' ]) || $elemkit_options[ 'settings' ][ 'ekit_user_consent_for_banner' ] === 'yes' ) {
			$elemkit_options [ 'settings' ][ 'ekit_user_consent_for_banner' ] = 'no';
		}

		update_option( 'elementskit_options', $elemkit_options );
	}

	public static function elementskit_menu() {
		remove_submenu_page( 'elementskit', 'forms' );
		remove_submenu_page( 'elementskit', 'elementskit-lite_wpmet_plugins' );
		remove_submenu_page( 'elementskit', 'elementskit-lite_get_help' );
	}

	public function elementskit_header_footer_body_class() {
		if (
			isset( $_GET['post'] ) &&
			in_array( get_post_type( $_GET['post'] ), [ 'elementskit_template', 'elementskit_widget', 'elementskit_content' ], true )
		) {
			?>
			<style>
				.elementor-control-hide_title,
				.elementor-control-hide_title ~ .elementor-control-separator {
					display: none !important;
				}
			</style>
			<?php
		}
	}

	public function disable_update_notifications() {
		add_filter( 'site_transient_update_plugins', [ $this, 'modify_elementskit_transient' ], 999 );
		// add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'modify_elementskit_transient' ], 999 );
	}

	public function modify_elementskit_transient( $transient ) {
		$plugin_file = 'elementskit/elementskit.php';

		$tgmpa_config = $this->get_tgmpa_plugin_config( 'elementskit' );
		
		if ( ! $tgmpa_config ) {
			return $transient;
		}

		$tgmpa_version = isset( $tgmpa_config['version'] ) ? $tgmpa_config['version'] : '';
		$tgmpa_package = isset( $tgmpa_config['source'] ) ? $tgmpa_config['source'] : '';

		if ( empty( $tgmpa_version ) || empty( $tgmpa_package ) ) {
			return $transient;
		}

		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file, false, false );
		$current_version = isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '';

		if ( ! empty( $current_version ) && version_compare( $current_version, $tgmpa_version, '=' ) ) {

			if ( ! is_object( $transient ) ) {
				$transient = new \stdClass();
			}
			if ( ! isset( $transient->response ) || ! is_array( $transient->response ) ) {
				$transient->response = array();
			}

			if ( isset( $transient->response[ $plugin_file ] ) ) {
				unset( $transient->response[ $plugin_file ] );
			}

			return $transient;
		}

		if ( ! is_object( $transient ) ) {
			$transient = new \stdClass();
		}
		if ( ! isset( $transient->response ) || ! is_array( $transient->response ) ) {
			$transient->response = array();
		}

		$plugin_name = isset( $plugin_data['Name'] ) ? $plugin_data['Name'] : 'ElementsKit';

		$download_url = '';
		if ( ! empty( $tgmpa_package ) ) {
			// Check if it's already a URL
			if ( filter_var( $tgmpa_package, FILTER_VALIDATE_URL ) ) {
				$download_url = $tgmpa_package;
			} else {
				// Convert file path to URL
				$content_dir = wp_normalize_path( WP_CONTENT_DIR );
				$normalized_package = wp_normalize_path( $tgmpa_package );
				
				if ( strpos( $normalized_package, $content_dir ) === 0 ) {
					// File is in wp-content directory
					$relative_path = str_replace( $content_dir, '', $normalized_package );
					$download_url = content_url( $relative_path );
				}
			}
		}

		$update_data = new \stdClass();
		// $update_data->id = 'elementskit/elementskit.php';
		$update_data->new_version = $tgmpa_version;
		$update_data->stable_version = $tgmpa_version;
		$update_data->name = $plugin_name;
		$update_data->slug = 'elementskit';
		$update_data->package = $tgmpa_package;
		// $update_data->download_link = $download_url;

		$transient->response[ $plugin_file ] = $update_data;

		return $transient;
	}

	private function get_tgmpa_plugin_config( $slug ) {

		if ( ! class_exists( 'TGM_Plugin_Activation' ) ) {
			return null;
		}

		$tgmpa_instance = call_user_func( array( get_class( $GLOBALS['tgmpa'] ), 'get_instance' ) );

		if ( ! $tgmpa_instance || ! isset( $tgmpa_instance->plugins ) ) {
			return null;
		}
		
		foreach ( $tgmpa_instance->plugins as $plugin ) {
			if ( isset( $plugin['slug'] ) && $plugin['slug'] === $slug ) {
				return $plugin;
			}
		}

		return null;
	}

}
KeyDesign_ElementsKit_Lite::instance();