<?php
/**
 * Theme Update Class
 * 
 * Handles theme update notifications and automatic updates.
 *
 * @package KeyDesign
 */

defined( 'ABSPATH' ) || exit;

class KeyDesign_Theme_Update {

	private $api_url = 'https://external.keydesign.xyz/';
	private $cache_key = 'keydesign_theme_versions';
	private $cache_duration = 3600;
	private static $instance = null;
	private $theme_data = null;

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->init_hooks();
	}

	private function init_hooks() {
		add_action( 'admin_notices', array( $this, 'display_update_notice' ) );
		add_action( 'network_admin_notices', array( $this, 'display_update_notice' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_update_scripts' ) );
		add_action( 'wp_ajax_keydesign_update_theme', array( $this, 'handle_theme_update_ajax' ) );
	}

	private function make_http_request( $url, $args = array() ) {
		$default_args = array(
			'timeout' => 15,
			'headers' => array(
				'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . esc_url_raw( home_url() )
			)
		);
		$args = wp_parse_args( $args, $default_args );

		$response = wp_remote_get( $url, $args );

		// Silently fail if connection error occurs
		if ( is_wp_error( $response ) ) {
			return null;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $response_code ) {
			return null;
		}

		$json_contents = wp_remote_retrieve_body( $response );
		$data = json_decode( $json_contents, true );

		// Return null if JSON is invalid or empty
		return ( null !== $data && is_array( $data ) ) ? $data : null;
	}

	private function get_theme_versions() {
		$cached_data = get_transient( $this->cache_key );
		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$data = $this->make_http_request( $this->api_url . 'versions.json' );
		if ( null === $data ) {
			return null;
		}

		set_transient( $this->cache_key, $data, $this->cache_duration );
		return $data;
	}

	private function get_current_theme() {
		if ( null === $this->theme_data ) {
			$theme = wp_get_theme();
			$this->theme_data = $theme->parent() ? $theme->parent() : $theme;
		}
		return $this->theme_data;
	}

	private function check_theme_version() {
		$theme = $this->get_current_theme();
		$theme_slug = strtolower( $theme->get( 'TextDomain' ) );
		$current_version = $theme->get( 'Version' );

		$versions = $this->get_theme_versions();
		
		if ( null === $versions || ! isset( $versions[ $theme_slug ] ) ) {
			return false;
		}
		
		$latest_version = $versions[ $theme_slug ];

		if ( version_compare( $current_version, $latest_version, '<' ) ) {
			return array(
				'current' => $current_version,
				'latest' => $latest_version,
				'theme_name' => $theme->get( 'Name' ),
				'theme_slug' => $theme_slug
			);
		}
		
		return false;
	}

	private function can_user_update_themes() {
		return current_user_can( 'update_themes' );
	}

	private function is_license_valid() {
		if ( ! class_exists( '\KeyDesign\License\Admin' ) || 
		     ! method_exists( '\KeyDesign\License\Admin', 'get_license_key' ) ) {
			return false;
		}

		$license_key = \KeyDesign\License\Admin::get_license_key();
		return ! empty( $license_key );
	}

	private function generate_action_button( $type, $args = array() ) {
		$defaults = array(
			'nonce' => '',
			'url' => '',
			'text' => '',
			'class' => 'button button-primary'
		);
		$args = wp_parse_args( $args, $defaults );

		switch ( $type ) {
			case 'update':
				return sprintf(
					'<button type="button" class="%s keydesign-update-btn" data-nonce="%s">%s</button>',
					esc_attr( $args['class'] ),
					esc_attr( $args['nonce'] ),
					esc_html( $args['text'] )
				);
			case 'register':
				return sprintf(
					'<a href="%s" class="%s">%s</a>',
					esc_url( $args['url'] ),
					esc_attr( $args['class'] ),
					esc_html( $args['text'] )
				);
			default:
				return '';
		}
	}

	private function generate_actions_section( $actions ) {
		if ( empty( $actions ) ) {
			return '';
		}

		$output = '<p class="keydesign-update-actions">';
		foreach ( $actions as $action ) {
			if ( isset( $action['message'] ) ) {
				$output .= '<em>' . esc_html( $action['message'] ) . '</em>';
				if ( count( $actions ) > 1 ) {
					$output .= '</p><p class="keydesign-update-actions">';
				}
			} elseif ( isset( $action['button'] ) ) {
				$output .= $action['button'];
			}
		}
		$output .= '</p>';
		return $output;
	}

	public function display_update_notice() {
		if ( ! class_exists( '\KeyDesign\Plugin' ) ) {
			return;
		}
		
		$update_info = $this->check_theme_version();
		
		if ( false === $update_info ) {
			return;
		}
		
		$nonce = wp_create_nonce( 'keydesign_update_theme' );
		
		$message = sprintf(
			/* translators: 1: theme name, 2: latest version */
			__( '<strong>%1$s theme update available!</strong> Version %2$s is now available. Update your theme to get the latest features, improvements, and security fixes.', 'sierra' ),
			esc_html( $update_info['theme_name'] ),
			esc_html( $update_info['latest'] )
		);
		
		$can_update = $this->can_user_update_themes() && $this->is_license_valid();
		
		echo '<div class="notice notice-error keydesign-update-notice">';
		echo '<p class="keydesign-update-message">' . wp_kses_post( $message ) . '</p>';
		
		$actions = array();
		
		if ( $can_update ) {
			$update_button = $this->generate_action_button( 'update', array(
				'nonce' => $nonce,
				'text' => __( 'Update now', 'sierra' )
			) );
			$actions[] = array( 'button' => $update_button . '<span class="spinner" style="float: none; margin: 0 0 0 10px; visibility: hidden;"></span>' );
		} else {
			if ( ! $this->is_license_valid() ) {
				$dashboard_url = admin_url( 'admin.php?page=keydesign-dashboard' );
				$actions[] = array( 'message' => __( 'Register your purchase to get access to auto-updates.', 'sierra' ) );
				$actions[] = array( 'button' => $this->generate_action_button( 'register', array(
					'url' => $dashboard_url,
					'text' => __( 'Register purchase', 'sierra' )
				) ) );
			} else {
				$actions[] = array( 'message' => __( 'You do not have permission to update themes.', 'sierra' ) );
			}
		}
		
		echo wp_kses_post( $this->generate_actions_section( $actions ) );
		
		echo '</div>';
	}

	public function enqueue_update_scripts() {
		$update_info = $this->check_theme_version();
		
		if ( false === $update_info ) {
			return;
		}

		$script = "
		jQuery(document).ready(function($) {
			$('.keydesign-update-btn').on('click', function(e) {
				e.preventDefault();
				
				var button = $(this);
				var notice = button.closest('.keydesign-update-notice');
				var spinner = notice.find('.spinner');
				var messageBox = notice.find('.keydesign-update-message');
				var nonce = button.data('nonce');
				
				// Disable button and show spinner
				button.prop('disabled', true);
				button.text('" . esc_js( __( 'Updating...', 'sierra' ) ) . "');
				spinner.css('visibility', 'visible');
				
				$.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'keydesign_update_theme',
						nonce: nonce
					},
					success: function(response) {
						spinner.css('visibility', 'hidden');
						
						if (response.success) {
							// Show success message
							notice.removeClass('notice-error').addClass('notice-success');
							messageBox.html('<strong>' + response.data.message + '</strong>');
							button.remove();
							
							// Reload page after 2 seconds
							setTimeout(function() {
								location.reload();
							}, 2000);
						} else {
							// Show error message
							messageBox.html('<strong>' + response.data.message + '</strong>');
							button.prop('disabled', false);
							button.text('" . esc_js( __( 'Update Now', 'sierra' ) ) . "');
						}
					},
					error: function(xhr, status, error) {
						spinner.css('visibility', 'hidden');
						messageBox.html('<strong>" . esc_js( __( 'An error occurred. Please try again or update manually.', 'sierra' ) ) . "</strong>');
						button.prop('disabled', false);
						button.text('" . esc_js( __( 'Update Now', 'sierra' ) ) . "');
					}
				});
			});
		});
		";
		
		wp_add_inline_script( 'jquery-core', $script );
	}

	public function handle_theme_update_ajax() {
		if ( ! $this->can_user_update_themes() ) {
			$this->send_json_error( __( 'You do not have permission to update themes.', 'sierra' ) );
		}

		if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'keydesign_update_theme' ) ) {
			$this->send_json_error( __( 'Security check failed. Please refresh the page and try again.', 'sierra' ) );
		}

		$theme_slug = get_template();
		
		$this->ensure_upgrade_files_loaded();

		$upgrader = new Theme_Upgrader( new WP_Ajax_Upgrader_Skin() );
		$result = $upgrader->upgrade( $theme_slug );
		
		if ( is_wp_error( $result ) ) {
			$this->send_json_error( sprintf(
				/* translators: %s: error message */
				__( 'Theme update failed: %s', 'sierra' ),
				$result->get_error_message()
			) );
		}
		
		if ( false === $result ) {
			$this->send_json_error( __( 'Theme update failed. Please try again or update manually.', 'sierra' ) );
		}

		wp_clean_themes_cache();
		delete_transient( $this->cache_key );
		
		$this->send_json_success( __( 'Theme updated successfully! The page will reload in 2 seconds.', 'sierra' ) );
	}

	private function ensure_upgrade_files_loaded() {
		$required_files = array(
			'class-wp-upgrader.php' => ABSPATH . 'wp-admin/includes/class-wp-upgrader.php',
			'wp_clean_themes_cache' => ABSPATH . 'wp-admin/includes/theme.php',
			'WP_Ajax_Upgrader_Skin' => ABSPATH . 'wp-admin/includes/file.php'
		);

		foreach ( $required_files as $check => $file ) {
			if ( 'wp_clean_themes_cache' === $check ) {
				if ( ! function_exists( $check ) ) {
					require_once $file;
				}
			} else {
				if ( ! class_exists( $check ) ) {
					require_once $file;
				}
			}
		}
	}

	private function send_json_error( $message ) {
		wp_send_json_error( array( 
			'message' => esc_html( $message ) 
		) );
	}
	
	private function send_json_success( $message ) {
		wp_send_json_success( array( 
			'message' => esc_html( $message )
		) );
	}

	public function clear_version_cache() {
		delete_transient( $this->cache_key );
	}
}
KeyDesign_Theme_Update::get_instance();